/*
 * Decompiled with CFR 0.152.
 */
package net.skinsrestorer.shared.storage;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.MalformedInputException;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.sql.rowset.CachedRowSet;
import net.skinsrestorer.api.SkinsRestorerAPI;
import net.skinsrestorer.api.exception.SkinRequestException;
import net.skinsrestorer.api.interfaces.ISkinStorage;
import net.skinsrestorer.api.property.GenericProperty;
import net.skinsrestorer.api.property.IProperty;
import net.skinsrestorer.shared.exception.NotPremiumException;
import net.skinsrestorer.shared.storage.Config;
import net.skinsrestorer.shared.storage.Locale;
import net.skinsrestorer.shared.storage.MySQL;
import net.skinsrestorer.shared.utils.C;
import net.skinsrestorer.shared.utils.connections.MineSkinAPI;
import net.skinsrestorer.shared.utils.connections.MojangAPI;
import net.skinsrestorer.shared.utils.log.SRLogger;

public class SkinStorage
implements ISkinStorage {
    private static final Pattern FORBIDDEN_CHARS_PATTERN = Pattern.compile("[\\\\/:*?\"<>|\\.]");
    private static final Pattern WHITESPACE_PATTERN = Pattern.compile("\\s");
    private static final String LTRIM = "^\\\\s+";
    private static final String RTRIM = "\\\\s+$";
    private static final Pattern TRIM_PATTERN = Pattern.compile("(^\\\\s+|\\\\s+$)");
    private final SRLogger logger;
    private final MojangAPI mojangAPI;
    private final MineSkinAPI mineSkinAPI;
    private MySQL mysql;
    private Path skinsFolder;
    private Path playersFolder;
    private boolean initialized = false;

    public void loadFolders(Path dataFolder) {
        this.skinsFolder = dataFolder.resolve("Skins");
        try {
            Files.createDirectories(this.skinsFolder, new FileAttribute[0]);
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        this.playersFolder = dataFolder.resolve("Players");
        try {
            Files.createDirectories(this.playersFolder, new FileAttribute[0]);
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    public void preloadDefaultSkins() {
        if (!Config.DEFAULT_SKINS_ENABLED) {
            return;
        }
        ArrayList toRemove = new ArrayList();
        Config.DEFAULT_SKINS.forEach(skin -> {
            try {
                if (!C.validUrl(skin)) {
                    this.getSkinForPlayer((String)skin);
                }
            }
            catch (SkinRequestException e) {
                toRemove.add(skin);
                this.logger.warning("[WARNING] DefaultSkin '" + skin + "'(.skin) could not be found or requested! Removing from list..");
                this.logger.debug("[DEBUG] DefaultSkin '" + skin + "' error: ", e);
            }
        });
        Config.DEFAULT_SKINS.removeAll(toRemove);
        if (Config.DEFAULT_SKINS.isEmpty()) {
            this.logger.warning("[WARNING] No more working DefaultSkin left... disabling feature");
            Config.DEFAULT_SKINS_ENABLED = false;
        }
    }

    public IProperty getDefaultSkinForPlayer(String playerName) throws SkinRequestException {
        String skin = this.getDefaultSkinName(playerName);
        if (C.validUrl(skin)) {
            return this.mineSkinAPI.genSkin(skin, null);
        }
        return this.getSkinForPlayer(skin);
    }

    @Override
    public IProperty getSkinForPlayer(String playerName) throws SkinRequestException {
        Optional<IProperty> textures;
        Optional<String> skin = this.getSkinOfPlayer(playerName);
        if (!skin.isPresent()) {
            skin = Optional.of(playerName.toLowerCase());
        }
        if (!(textures = this.getSkinData(skin.get())).isPresent()) {
            try {
                if (!C.validMojangUsername(skin.get())) {
                    throw new SkinRequestException(Locale.INVALID_PLAYER.replace("%player", skin.get()));
                }
                textures = this.mojangAPI.getSkin(skin.get());
                if (!textures.isPresent()) {
                    throw new SkinRequestException(Locale.ERROR_NO_SKIN);
                }
                this.setSkinData(skin.get(), textures.get());
                return textures.get();
            }
            catch (SkinRequestException e) {
                throw e;
            }
            catch (Exception e) {
                e.printStackTrace();
                throw new SkinRequestException(Locale.WAIT_A_MINUTE);
            }
        }
        return textures.get();
    }

    @Override
    public Optional<String> getSkinOfPlayer(String playerName) {
        playerName = playerName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT * FROM " + Config.MYSQL_PLAYER_TABLE + " WHERE Nick=?", playerName);
            if (crs != null) {
                try {
                    String skin = crs.getString("Skin");
                    if (skin.isEmpty()) {
                        this.removeSkinOfPlayer(playerName);
                        return Optional.empty();
                    }
                    return Optional.of(skin);
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
            }
        } else {
            playerName = this.replaceForbiddenChars(playerName);
            Path playerFile = this.playersFolder.resolve(playerName + ".player");
            try {
                if (!Files.exists(playerFile, new LinkOption[0])) {
                    return Optional.empty();
                }
                List<String> lines = Files.readAllLines(playerFile);
                if (lines.isEmpty()) {
                    this.removeSkinOfPlayer(playerName);
                    return Optional.empty();
                }
                return Optional.of(lines.get(0));
            }
            catch (MalformedInputException e) {
                this.removeSkinOfPlayer(playerName);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
        return Optional.empty();
    }

    private IProperty createProperty(String playerName, boolean updateOutdated, String value, String signature, long timestamp) throws SkinRequestException {
        Optional<IProperty> skin;
        if (updateOutdated && C.validMojangUsername(playerName) && this.isExpired(timestamp) && (skin = this.mojangAPI.getSkin(playerName)).isPresent()) {
            this.setSkinData(playerName, skin.get());
            return skin.get();
        }
        return SkinsRestorerAPI.getApi().createPlatformProperty("textures", value, signature);
    }

    @Override
    public void removeSkinOfPlayer(String playerName) {
        playerName = playerName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("DELETE FROM " + Config.MYSQL_PLAYER_TABLE + " WHERE Nick=?", playerName);
        } else {
            playerName = this.replaceForbiddenChars(playerName);
            Path playerFile = this.playersFolder.resolve(playerName + ".player");
            try {
                Files.deleteIfExists(playerFile);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void setSkinOfPlayer(String playerName, String skinName) {
        playerName = playerName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("INSERT INTO " + Config.MYSQL_PLAYER_TABLE + " (Nick, Skin) VALUES (?,?) ON DUPLICATE KEY UPDATE Skin=?", playerName, skinName, skinName);
        } else {
            playerName = this.replaceForbiddenChars(playerName);
            Path playerFile = this.playersFolder.resolve(playerName + ".player");
            try (OutputStreamWriter writer = new OutputStreamWriter(Files.newOutputStream(playerFile, new OpenOption[0]), StandardCharsets.UTF_8);){
                skinName = this.removeWhitespaces(skinName);
                skinName = this.replaceForbiddenChars(skinName);
                writer.write(skinName);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public Optional<IProperty> getSkinData(String skinName) {
        return this.getSkinData(skinName, true);
    }

    @Override
    public Optional<IProperty> getSkinData(String skinName, boolean updateOutdated) {
        skinName = skinName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT * FROM " + Config.MYSQL_SKIN_TABLE + " WHERE Nick=?", skinName);
            if (crs == null) {
                return Optional.empty();
            }
            try {
                String value = crs.getString("Value");
                String signature = crs.getString("Signature");
                String timestamp = crs.getString("timestamp");
                return Optional.of(this.createProperty(skinName, updateOutdated, value, signature, Long.parseLong(timestamp)));
            }
            catch (Exception e) {
                this.logger.info(String.format("Unsupported skin format.. removing (%s).", skinName));
                this.removeSkinData(skinName);
            }
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.replaceForbiddenChars(skinName);
            Path skinFile = this.skinsFolder.resolve(skinName + ".skin");
            try {
                if (!Files.exists(skinFile, new LinkOption[0])) {
                    return Optional.empty();
                }
                List<String> lines = Files.readAllLines(skinFile);
                String value = lines.get(0);
                String signature = lines.get(1);
                String timestamp = lines.get(2);
                return Optional.of(this.createProperty(skinName, updateOutdated, value, signature, Long.parseLong(timestamp)));
            }
            catch (Exception e) {
                this.logger.info(String.format("Unsupported skin format.. removing (%s).", skinName));
                this.removeSkinData(skinName);
            }
        }
        return Optional.empty();
    }

    public void removeSkinData(String skinName) {
        skinName = skinName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("DELETE FROM " + Config.MYSQL_SKIN_TABLE + " WHERE Nick=?", skinName);
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.replaceForbiddenChars(skinName);
            Path skinFile = this.skinsFolder.resolve(skinName + ".skin");
            try {
                Files.deleteIfExists(skinFile);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void setSkinData(String skinName, IProperty textures) {
        this.setSkinData(skinName, textures, System.currentTimeMillis());
    }

    @Override
    public void setSkinData(String skinName, IProperty textures, long timestamp) {
        skinName = skinName.toLowerCase();
        String value = textures.getValue();
        String signature = textures.getSignature();
        String timestampString = Long.toString(timestamp);
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("INSERT INTO " + Config.MYSQL_SKIN_TABLE + " (Nick, Value, Signature, timestamp) VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE Value=?, Signature=?, timestamp=?", skinName, value, signature, timestampString, value, signature, timestampString);
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.replaceForbiddenChars(skinName);
            Path skinFile = this.skinsFolder.resolve(skinName + ".skin");
            try {
                if (value.isEmpty() || signature.isEmpty() || timestampString.isEmpty()) {
                    return;
                }
                try (OutputStreamWriter writer = new OutputStreamWriter(Files.newOutputStream(skinFile, new OpenOption[0]), StandardCharsets.UTF_8);){
                    writer.write(value + "\n" + signature + "\n" + timestamp);
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public Map<String, IProperty> getSkins(int number) {
        TreeMap<String, IProperty> list = new TreeMap<String, IProperty>();
        if (Config.MYSQL_ENABLED) {
            String filterBy = "";
            String orderBy = "Nick";
            if (Config.CUSTOM_GUI_ENABLED) {
                if (Config.CUSTOM_GUI_ONLY) {
                    filterBy = "WHERE Nick RLIKE '" + String.join((CharSequence)"|", Config.CUSTOM_GUI_SKINS) + "'";
                } else {
                    orderBy = "FIELD(Nick, " + Config.CUSTOM_GUI_SKINS.stream().map(skin -> "'" + skin + "'").collect(Collectors.joining(", ")) + ") DESC, Nick";
                }
            }
            CachedRowSet crs = this.mysql.query("SELECT Nick, Value, Signature FROM " + Config.MYSQL_SKIN_TABLE + " " + filterBy + " ORDER BY " + orderBy, new Object[0]);
            int i = 0;
            try {
                do {
                    if (i >= number) {
                        list.put(crs.getString("Nick").toLowerCase(), SkinsRestorerAPI.getApi().createPlatformProperty("textures", crs.getString("Value"), crs.getString("Signature")));
                    }
                    ++i;
                } while (crs.next());
            }
            catch (SQLException sQLException) {}
        } else {
            ArrayList files = new ArrayList();
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.skinsFolder, "*.skin");){
                stream.forEach(files::add);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            List skinNames = files.stream().map(Path::getFileName).map(Path::toString).map(s -> s.substring(0, s.length() - 5)).sorted().collect(Collectors.toList());
            int i = 0;
            for (String skinName : skinNames) {
                if (i >= number) {
                    if (Config.CUSTOM_GUI_ONLY) {
                        for (String GuiSkins : Config.CUSTOM_GUI_SKINS) {
                            if (!skinName.toLowerCase().contains(GuiSkins.toLowerCase())) continue;
                            this.getSkinData(skinName, false).ifPresent(property -> list.put(skinName.toLowerCase(), (IProperty)property));
                        }
                    } else {
                        this.getSkinData(skinName, false).ifPresent(property -> list.put(skinName.toLowerCase(), (IProperty)property));
                    }
                }
                ++i;
            }
        }
        return list;
    }

    public Map<String, GenericProperty> getSkinsRaw(int number) {
        TreeMap<String, GenericProperty> list = new TreeMap<String, GenericProperty>();
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT Nick, Value, Signature FROM " + Config.MYSQL_SKIN_TABLE + " ORDER BY `Nick`", new Object[0]);
            int i = 0;
            int foundSkins = 0;
            try {
                do {
                    if (i >= number && foundSkins <= 25) {
                        GenericProperty prop = new GenericProperty();
                        prop.setName("textures");
                        prop.setValue(crs.getString("Value"));
                        prop.setSignature(crs.getString("Signature"));
                        list.put(crs.getString("Nick"), prop);
                        ++foundSkins;
                    }
                    ++i;
                } while (crs.next());
            }
            catch (SQLException sQLException) {}
        } else {
            ArrayList<Path> files = new ArrayList<Path>();
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.skinsFolder, "*.skin");){
                stream.forEach(files::add);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            files.sort(Comparator.comparing(o -> o.getFileName().toString()));
            int i = 0;
            int foundSkins = 0;
            for (Path file : files) {
                String fileName = file.getFileName().toString();
                String skinName = fileName.substring(0, fileName.length() - 5);
                if (i >= number && foundSkins <= 25) {
                    try {
                        if (!Files.exists(file, new LinkOption[0])) continue;
                        List<String> lines = Files.readAllLines(file);
                        GenericProperty prop = new GenericProperty();
                        prop.setName("textures");
                        prop.setValue(lines.get(0));
                        prop.setSignature(lines.get(1));
                        list.put(skinName, prop);
                        ++foundSkins;
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
                ++i;
            }
        }
        return list;
    }

    public boolean updateSkinData(String skinName) throws SkinRequestException {
        if (!C.validMojangUsername(skinName)) {
            throw new SkinRequestException(Locale.ERROR_UPDATING_CUSTOMSKIN);
        }
        boolean updateDisabled = false;
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT timestamp FROM " + Config.MYSQL_SKIN_TABLE + " WHERE Nick=?", skinName);
            if (crs != null) {
                try {
                    updateDisabled = crs.getString("timestamp").equals("0");
                }
                catch (Exception exception) {}
            }
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.replaceForbiddenChars(skinName);
            Path skinFile = this.skinsFolder.resolve(skinName + ".skin");
            try {
                if (Files.exists(skinFile, new LinkOption[0])) {
                    updateDisabled = Files.readAllLines(skinFile).get(2).equals("0");
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        if (updateDisabled) {
            throw new SkinRequestException(Locale.ERROR_UPDATING_CUSTOMSKIN);
        }
        try {
            Optional<IProperty> textures;
            Optional<String> mojangUUID = this.mojangAPI.getUUIDMojang(skinName);
            if (mojangUUID.isPresent() && (textures = this.mojangAPI.getProfileMojang(mojangUUID.get())).isPresent()) {
                this.setSkinData(skinName, textures.get());
                return true;
            }
        }
        catch (NotPremiumException e) {
            throw new SkinRequestException(Locale.ERROR_UPDATING_CUSTOMSKIN);
        }
        return false;
    }

    public String getDefaultSkinName(String playerName) {
        return this.getDefaultSkinName(playerName, false);
    }

    public String getDefaultSkinName(String playerName, boolean clear) {
        playerName = TRIM_PATTERN.matcher(playerName).replaceAll("");
        if (Config.DEFAULT_SKINS_ENABLED) {
            if (!Config.DEFAULT_SKINS_PREMIUM) {
                try {
                    if (C.validMojangUsername(playerName) && this.mojangAPI.getUUID(playerName) != null) {
                        return playerName;
                    }
                }
                catch (SkinRequestException skinRequestException) {
                    // empty catch block
                }
            }
            if (!this.getSkinOfPlayer(playerName).isPresent() || clear) {
                List<String> skins = Config.DEFAULT_SKINS;
                String randomSkin = skins.size() > 1 ? skins.get(new Random().nextInt(skins.size())) : skins.get(0);
                return randomSkin != null ? randomSkin : playerName;
            }
        }
        if (clear) {
            return playerName;
        }
        return this.getSkinOfPlayer(playerName).orElse(playerName);
    }

    private boolean isExpired(long timestamp) {
        if (timestamp == 0L || Config.DISALLOW_AUTO_UPDATE_SKIN) {
            return false;
        }
        return timestamp + TimeUnit.MINUTES.toMillis(Config.SKIN_EXPIRES_AFTER) <= System.currentTimeMillis();
    }

    private String replaceForbiddenChars(String str) {
        return FORBIDDEN_CHARS_PATTERN.matcher(str).replaceAll("\u00b7");
    }

    private String removeWhitespaces(String str) {
        if (str.startsWith(" ")) {
            return str;
        }
        return WHITESPACE_PATTERN.matcher(str).replaceAll("");
    }

    public boolean purgeOldSkins(int days) {
        boolean bl;
        block12: {
            long targetPurgeTimestamp = System.currentTimeMillis() - (long)days * 86400L * 1000L;
            if (Config.MYSQL_ENABLED) {
                this.mysql.execute("DELETE FROM " + Config.MYSQL_SKIN_TABLE + " WHERE Nick NOT LIKE ' %' AND " + Config.MYSQL_SKIN_TABLE + ".timestamp NOT LIKE 0 AND " + Config.MYSQL_SKIN_TABLE + ".timestamp<=?", targetPurgeTimestamp);
                return true;
            }
            DirectoryStream<Path> stream = Files.newDirectoryStream(this.skinsFolder, "*.skin");
            try {
                for (Path file : stream) {
                    try {
                        List<String> lines;
                        Long timestamp;
                        if (!Files.exists(file, new LinkOption[0]) || (timestamp = Long.valueOf((lines = Files.readAllLines(file)).get(2))).equals(0L) || timestamp >= targetPurgeTimestamp) continue;
                        Files.deleteIfExists(file);
                    }
                    catch (Exception exception) {}
                }
                bl = true;
                if (stream == null) break block12;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    e.printStackTrace();
                    return false;
                }
            }
            stream.close();
        }
        return bl;
    }

    public SkinStorage(SRLogger logger, MojangAPI mojangAPI, MineSkinAPI mineSkinAPI) {
        this.logger = logger;
        this.mojangAPI = mojangAPI;
        this.mineSkinAPI = mineSkinAPI;
    }

    public void setMysql(MySQL mysql) {
        this.mysql = mysql;
    }

    public void setInitialized(boolean initialized) {
        this.initialized = initialized;
    }

    @Override
    public boolean isInitialized() {
        return this.initialized;
    }
}

